// MINTOS INJECTOR v4.0 - Licensed Edition
const axios = require('axios');
const { io } = require('socket.io-client');
const express = require('express');
const http = require('http');
const fs = require('fs');
const os = require('os');
const path = require('path');
const crypto = require('crypto');

const BASE = 'https://tr.minticity.com';
const SOCKET_URL = 'https://socket.minticity.com:8080';
const DASH_PORT = 4000;
const KEY_SERVER = 'http://multiversesmp.com:3778';

// CLI args: node inject.js [key] [username] [password] [connections]
const LICENSE_KEY = process.argv[2] || '';
let USER = process.argv[3] || '';
let PASS = process.argv[4] || '';
const TARGET = parseInt(process.argv[5]) || 50;

// Generate hardware ID for key binding
function getHWID() {
  const cpu = os.cpus()[0]?.model || '';
  const hostname = os.hostname();
  const platform = os.platform() + os.arch();
  const totalMem = os.totalmem().toString();
  return crypto.createHash('sha256').update(cpu + hostname + platform + totalMem).digest('hex').substring(0, 16);
}

// License limits (populated after validation)
const keyLimits = { expiresAt: null, mintosLimit: null, mintosUsed: 0 };

// Validate license key against VPS
async function validateKey(key) {
  if (!key) throw new Error('No license key provided! Get one from the admin.');
  const hwid = getHWID();
  try {
    const res = await axios.post(`${KEY_SERVER}/validate`, { key, hwid }, { timeout: 10000 });
    if (!res.data.valid) throw new Error(res.data.error || 'Invalid key');
    if (res.data.expiresAt) keyLimits.expiresAt = new Date(res.data.expiresAt);
    if (res.data.mintosLimit) keyLimits.mintosLimit = res.data.mintosLimit;
    if (res.data.mintosUsed) keyLimits.mintosUsed = res.data.mintosUsed;
    return true;
  } catch (e) {
    if (e.response?.data?.error) throw new Error(e.response.data.error);
    if (e.code === 'ECONNREFUSED') throw new Error('Key server offline - contact admin');
    throw e;
  }
}

// Report mintos to key server and check limits
let lastReported = 0;
async function reportMintos() {
  const delta = state.mintos - lastReported;
  if (delta <= 0) return;
  try {
    const res = await axios.post(`${KEY_SERVER}/report`, {
      key: LICENSE_KEY, hwid: getHWID(), mintos: delta
    }, { timeout: 5000 });
    if (res.data.ok) {
      lastReported = state.mintos;
      if (res.data.limitReached) {
        log('MINTOS LIMIT REACHED! Stopping...');
        state.running = false;
      }
      if (res.data.expired) {
        log('KEY EXPIRED! Stopping...');
        state.running = false;
      }
    }
  } catch {}
}

function checkKeyExpiry() {
  if (keyLimits.expiresAt && Date.now() > keyLimits.expiresAt.getTime()) {
    log('KEY EXPIRED! Stopping...');
    state.running = false;
    return true;
  }
  if (keyLimits.mintosLimit && (state.mintos + keyLimits.mintosUsed) >= keyLimits.mintosLimit) {
    log('MINTOS LIMIT REACHED! Stopping...');
    state.running = false;
    return true;
  }
  return false;
}
const BATCH = 1000;
const BATCH_DELAY = 2000;
const STAGGER = 50;
const MIN_FREE_MB = 2048;
const DB_SAVE_EVERY = 50;

let answerDB = {};
try { answerDB = JSON.parse(fs.readFileSync('./answer-db.json', 'utf8')); } catch {}

const state = {
  running: false, conns: 0, active: 0,
  games: 0, mintos: 0, correct: 0, wrong: 0, questions: 0,
  errors: 0, startTime: 0, cookies: '', words: Object.keys(answerDB).length,
  log: [], user: USER,
};

const delay = ms => new Promise(r => setTimeout(r, ms));
const freeMB = () => Math.round(os.freemem() / 1024 / 1024);
const totalMB = () => Math.round(os.totalmem() / 1024 / 1024);

function log(msg) {
  const line = `[${new Date().toLocaleTimeString()}] ${msg}`;
  console.log(line);
  state.log.push(line);
  if (state.log.length > 200) state.log.shift();
}

// === LOGIN ===
async function login() {
  const jar = {};
  const ua = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36';
  function save(h) {
    (h['set-cookie'] || []).forEach(c => {
      const eq = c.indexOf('='), semi = c.indexOf(';');
      if (eq > 0) jar[c.substring(0, eq).trim()] = c.substring(eq + 1, semi > 0 ? semi : undefined).trim();
    });
  }
  function cs() { return Object.entries(jar).map(([k, v]) => `${k}=${v}`).join('; '); }

  log(`Fetching login page...`);
  const r1 = await axios.get(`${BASE}/login`, { headers: { 'User-Agent': ua }, validateStatus: () => true });
  save(r1.headers);
  log(`Login page: ${r1.status}`);
  const token = (r1.data.match(/name="_token"[^>]*value="([^"]+)"/) || [])[1]
    || (r1.data.match(/value="([^"]+)"[^>]*name="_token"/) || [])[1]
    || (r1.data.match(/name="csrf-token"\s+content="([^"]+)"/) || [])[1]
    || (r1.data.match(/content="([^"]+)"\s+name="csrf-token"/) || [])[1]
    || (r1.data.match(/_token.*?value="([^"]+)"/) || [])[1]
    || (r1.data.match(/"_token"\s*:\s*"([^"]+)"/) || [])[1]
    || '';
  if (!token) {
    log('WARNING: No CSRF token found! Login will likely fail.');
    // Show a snippet around _token or csrf to debug
    const idx = r1.data.indexOf('_token');
    if (idx >= 0) log('TOKEN HINT: ' + r1.data.substring(idx - 30, idx + 80).replace(/\s+/g, ' '));
    else log('No _token string found in page at all');
  } else {
    log(`CSRF token found (${token.length} chars)`);
  }
  const xsrf = jar['XSRF-TOKEN'] ? decodeURIComponent(jar['XSRF-TOKEN']) : '';

  log(`Sending login for "${USER}" (${Buffer.from(USER).length} bytes)...`);
  const r2 = await axios.post(`${BASE}/login`,
    new URLSearchParams({ _token: token, username: USER, password: PASS }).toString(), {
      headers: { 'User-Agent': ua, 'Cookie': cs(), 'X-XSRF-TOKEN': xsrf, 'Content-Type': 'application/x-www-form-urlencoded', 'Referer': `${BASE}/login`, 'Origin': BASE },
      maxRedirects: 0, validateStatus: () => true,
    });
  save(r2.headers);
  log(`Login POST: ${r2.status}`);

  if (r2.status === 200) {
    // Check if it's a JS/meta redirect (login success with different redirect style)
    const jsRedirect = r2.data.match(/(?:window\.location|location\.href|url)\s*[=:]\s*["']([^"']*junior[^"']*)/i) ||
                       r2.data.match(/https?:\/\/[^"'\s]*junior\/index/i);
    if (jsRedirect || r2.data.includes('/junior/index') || r2.data.includes('junior')) {
      log('Login OK (JS redirect detected)');
      // Follow the redirect manually
      const r3 = await axios.get(`${BASE}/junior/index`, { headers: { 'User-Agent': ua, 'Cookie': cs() }, maxRedirects: 5, validateStatus: () => true });
      save(r3.headers);
    } else {
      // Actually failed - wrong credentials
      const errMatch = r2.data.match(/class="[^"]*error[^"]*"[^>]*>([^<]+)/i) ||
                       r2.data.match(/class="[^"]*alert[^"]*"[^>]*>([^<]+)/i) ||
                       r2.data.match(/class="[^"]*invalid[^"]*"[^>]*>([^<]+)/i) ||
                       r2.data.match(/class="[^"]*danger[^"]*"[^>]*>([^<]+)/i);
      const errMsg = errMatch ? errMatch[1].trim() : 'Unknown reason';
      throw new Error(`Invalid username or password! Server said: ${errMsg}`);
    }
  }
  if (r2.status === 404) {
    throw new Error('Login endpoint returned 404 - account may not exist');
  }
  if (r2.status === 419) {
    throw new Error('CSRF token expired (419) - try again');
  }
  if (r2.status === 429) {
    throw new Error('Rate limited (429) - wait a few minutes and try again');
  }

  if (r2.status >= 300 && r2.status < 400) {
    const loc = r2.headers.location?.startsWith('http') ? r2.headers.location : `${BASE}${r2.headers.location}`;
    log(`Redirect -> ${loc}`);
    const r3 = await axios.get(loc, { headers: { 'User-Agent': ua, 'Cookie': cs() }, maxRedirects: 5, validateStatus: () => true });
    save(r3.headers);
  }

  log('Loading game page...');
  const r4 = await axios.get(`${BASE}/junior/spiel-spass/der-die-das`, {
    headers: { 'User-Agent': ua, 'Cookie': cs() }, validateStatus: () => true });
  save(r4.headers);
  log(`Game page: ${r4.status} | socket: ${r4.data.includes('socket')} | phaser: ${r4.data.includes('phaser')}`);
  if (!r4.data.includes('socket') && !r4.data.includes('phaser')) throw new Error('Login succeeded but game page has no game content - account may not have access');
  return cs();
}

// === SOCKET / GAME (exact same as proven inject-terminal.js) ===
function connectSocket(cookies) {
  return new Promise((resolve, reject) => {
    const s = io(SOCKET_URL, {
      transports: ['websocket'],
      extraHeaders: { 'Cookie': cookies, 'Origin': BASE },
      reconnection: false, timeout: 15000, forceNew: true,
    });
    s.on('connect', () => resolve(s));
    s.on('connect_error', () => { try { s.disconnect(); } catch {} reject(new Error('ce')); });
    s.on('error', () => {});
    setTimeout(() => { try { s.disconnect(); } catch {} reject(new Error('to')); }, 15000);
  });
}

function playGame(socket) {
  return new Promise((resolve) => {
    let c = 0, w = 0, q = 0, ended = false;
    function finish(cr = 0) { if (ended) return; ended = true; resolve({ c, w, q, cr }); }
    function nextQ() {
      if (q >= 200 || ended) return endG();
      const t = setTimeout(() => endG(), 3000);
      socket.once('dddSetQuestion', (word) => {
        clearTimeout(t);
        if (!word || ended) return endG();
        const ans = answerDB[word] || (Math.random() < 0.45 ? 'die' : Math.random() < 0.65 ? 'der' : 'das');
        const t2 = setTimeout(() => endG(), 3000);
        socket.once('dddControlAnswer', (data) => {
          clearTimeout(t2);
          if (!data || ended) return endG();
          if (data.correctAnswer) answerDB[word] = data.correctAnswer;
          data.isCorrect ? c++ : w++;
          q++;
          data.isEnded ? endG() : nextQ();
        });
        socket.emit('dddControlQuestion', ans);
      });
      socket.emit('dddGetQuestion');
    }
    function endG() {
      if (ended) return;
      socket.once('updateCredit', (amt) => finish(amt));
      socket.emit('dddEndGame');
      setTimeout(() => finish(0), 5000);
    }
    socket.once('dddSetStart', nextQ);
    socket.once('dddSetStarting', () => socket.emit('dddCreate', { category: 15, level: 3 }));
    socket.emit('dddInitialize');
    setTimeout(() => { if (!ended) endG(); }, 45000);
  });
}

// === CONNECTION LOOP (exact same as proven inject-terminal.js) ===
async function connectionLoop() {
  state.conns++;
  let errStreak = 0;
  while (state.running) {
    let socket;
    try {
      socket = await connectSocket(state.cookies);
      errStreak = 0;
      while (state.running) {
        state.active++;
        const r = await playGame(socket);
        state.active--;
        if (r.q > 0) {
          state.games++;
          state.mintos += r.cr;
          state.correct += r.c;
          state.wrong += r.w;
          state.questions += r.q;
          state.words = Object.keys(answerDB).length;
          if (state.games % DB_SAVE_EVERY === 0) {
            try { fs.writeFileSync('./answer-db.json', JSON.stringify(answerDB, null, 2)); } catch {}
          }
        }
        if (!state.running) break;
        const ok = await new Promise(res => {
          socket.once('connect', () => res(true));
          socket.disconnect();
          setTimeout(() => socket.connect(), 100);
          setTimeout(() => res(false), 5000);
        });
        if (!ok) break;
      }
    } catch {
      errStreak++;
      state.errors++;
      await delay(errStreak > 5 ? 5000 : 1000);
    } finally {
      try { socket?.disconnect(); } catch {}
    }
  }
  state.conns--;
}

function fmtNum(n) {
  if (n >= 1e9) return (n / 1e9).toFixed(2) + 'B';
  if (n >= 1e6) return (n / 1e6).toFixed(2) + 'M';
  if (n >= 1e3) return (n / 1e3).toFixed(1) + 'K';
  return n.toLocaleString();
}

function printStats() {
  const el = (Date.now() - state.startTime) / 1000;
  const mpm = el > 0 ? (state.mintos / el) * 60 : 0;
  const mph = mpm * 60;
  const gpm = el > 0 ? (state.games / el) * 60 : 0;
  const acc = state.questions > 0 ? ((state.correct / state.questions) * 100).toFixed(1) : '0.0';
  const h = Math.floor(el / 3600), m = Math.floor((el % 3600) / 60), s = Math.floor(el % 60);
  const mem = Math.round(process.memoryUsage().rss / 1024 / 1024);

  log(`CONNS ${state.conns}/${TARGET} | ACTIVE ${state.active} | MINTOS ${fmtNum(state.mintos)} | /min ${fmtNum(Math.round(mpm))} | /hr ${fmtNum(Math.round(mph))} | /day ${fmtNum(Math.round(mph * 24))} | GAMES ${state.games} (${gpm.toFixed(0)}/m) | ACC ${acc}% | ERR ${state.errors} | RAM ${mem}MB | ${h}h${m}m${s}s`);
}

// === STATS DASHBOARD (added on top) ===
function startDashboard() {
  const app = express();
  const server = http.createServer(app);

  app.get('/', (req, res) => res.sendFile(path.join(__dirname, 'inject-stats.html')));

  app.get('/api/stats', (req, res) => {
    const el = state.running ? (Date.now() - state.startTime) / 1000 : 0;
    const mpm = el > 0 ? (state.mintos / el) * 60 : 0;
    const mph = mpm * 60;
    const gpm = el > 0 ? (state.games / el) * 60 : 0;
    const acc = state.questions > 0 ? ((state.correct / state.questions) * 100).toFixed(1) : '0.0';
    const mem = process.memoryUsage();
    res.json({
      running: state.running, target: TARGET, user: USER,
      conns: state.conns, active: state.active, uptime: el,
      mintos: state.mintos, mintosPerMin: Math.round(mpm),
      mintosPerHour: Math.round(mph), mintosPerDay: Math.round(mph * 24),
      games: state.games, gamesPerMin: parseFloat(gpm.toFixed(1)),
      questions: state.questions, accuracy: acc, words: state.words,
      errors: state.errors, ramUsed: Math.round(mem.rss / 1024 / 1024),
      ramFree: freeMB(), ramTotal: totalMB(), cpu: os.loadavg()[0].toFixed(2),
      log: state.log.slice(-50),
    });
  });

  server.listen(DASH_PORT, () => {
    log(`Dashboard: http://localhost:${DASH_PORT}`);
  });
}

// Crash protection
process.on('uncaughtException', (e) => { log(`UNCAUGHT: ${e.message}`); });
process.on('unhandledRejection', (e) => { log(`UNHANDLED: ${e?.message || e}`); });

// === MAIN ===
async function main() {
  log(`MINTOS INJECTOR v4.0 - Licensed Edition`);
  log(`User: ${USER} | Target: ${TARGET} connections`);

  // Validate license key before anything else
  log('Validating license key...');
  await validateKey(LICENSE_KEY);
  log('License key valid!');
  if (keyLimits.expiresAt) log(`Key expires: ${keyLimits.expiresAt.toLocaleString()}`);
  if (keyLimits.mintosLimit) log(`Mintos limit: ${keyLimits.mintosLimit.toLocaleString()} (${keyLimits.mintosUsed.toLocaleString()} used)`);

  log('Logging in...');
  state.cookies = await login();
  log('Login OK!');

  // Start dashboard + open browser
  startDashboard();
  const { exec } = require('child_process');
  exec(`start http://localhost:${DASH_PORT}`);

  state.running = true;
  state.startTime = Date.now();

  // Print stats every 2 seconds, report mintos every 30 seconds
  setInterval(printStats, 2000);
  setInterval(() => { reportMintos(); checkKeyExpiry(); }, 30000);

  log(`Launching ${TARGET} connections...`);
  for (let i = 0; i < TARGET; i++) {
    if (!state.running) break;
    if (i > 0 && i % BATCH === 0) {
      const free = freeMB();
      if (free < MIN_FREE_MB) { log(`Low memory (${free}MB), stopped at ${i}`); break; }
      printStats();
      await delay(BATCH_DELAY);
    } else if (i > 0) {
      await delay(STAGGER);
    }
    connectionLoop();
  }
  log('All connections launched!');

  process.on('SIGINT', () => {
    log('Shutting down...');
    state.running = false;
    try { fs.writeFileSync('./answer-db.json', JSON.stringify(answerDB, null, 2)); } catch {}
    printStats();
    setTimeout(() => process.exit(0), 3000);
  });
}

main().catch(e => { console.error('Fatal:', e.message); process.exit(1); });
